# scripts/calibrate_pivot.py
#!/usr/bin/env python3
"""
Calibrate pivot parameters (a, b) for the affine pivot function g(D) = a*D + b,
using two reference points:
  g(D_ref1) = g_ref1 and g(D_ref2) = g_ref2.
Reads a fitted fractal curve (unused here, but kept for provenance) and
writes pivot_params.json.
"""
import argparse
import pickle
import json


def main():
    parser = argparse.ArgumentParser(
        description="Calibrate pivot parameters for g(D) = a*D + b"
    )
    parser.add_argument(
        "fit_pickle", help="Path to pickle file containing fractal-fit parameters"
    )
    parser.add_argument(
        "output_json", help="Path to write pivot_params.json"
    )
    args = parser.parse_args()

    # Load fractal-fit parameters (for provenance only)
    try:
        with open(args.fit_pickle, "rb") as f:
            popt = pickle.load(f)
    except Exception:
        popt = None  # no-op if unavailable

    # Reference points for calibration
    D_ref1, g_ref1 = 2.0, 1.0    # Earth gravity normalization
    D_ref2, g_ref2 = 2.20, 0.73  # Cosmic weakening normalization

    # Solve for a and b in g(D) = a*D + b
    a = (g_ref2 - g_ref1) / (D_ref2 - D_ref1)
    b = g_ref1 - a * D_ref1

    pivot = {"a": a, "b": b}
    # Write output
    with open(args.output_json, "w") as f:
        json.dump(pivot, f, indent=2)

    print(f"Calibrated pivot parameters: a={a:.6f}, b={b:.6f}")


if __name__ == "__main__":
    main()
